/*
 *	psim.c
 *	Frank Neubauer, 1994
 *
 *	This module contains the fuzzy-simulationloop.
 */

#include <stdio.h>
#include <string.h>
#include "psim.h"

#define	HEADER_SIZE		(9*4)

typedef struct {
	VALUE	x,y;
	double		d,s;
} Point;

typedef enum {
	OP_END = 0x00, 
	OP_CLEAR = 0x01,
	OP_FUZZYFYINT = 0x10, 
	OP_DEFUZZYINT_COG = 0x20, 
	OP_DEFUZZYINT_MHL = 0x21, 
	OP_DEFUZZYINT_MHR = 0x22,
	OP_WEIGHT = 0x30,
	OP_MIN = 0x31,
	OP_MAX = 0x32
} OPCODE;

typedef BYTE*	(*CODEPROC)(BYTE*);

static BYTE*	OP_Clear( BYTE * pMem );
static BYTE*	OP_FuzzifyInt( BYTE * pMem );
static BYTE*	OP_Weight( BYTE * pMem );
static BYTE*	OP_Min( BYTE * pMem );
static BYTE*	OP_Max( BYTE * pMem );
static BYTE*	OP_DefuzzyIntCOG( BYTE * pMem );
static BYTE*	OP_DefuzzyIntMHL( BYTE * pMem );
static BYTE*	OP_DefuzzyIntMHR( BYTE * pMem );

typedef struct {
	OPCODE		Opcode;
	CODEPROC	CodeProc;
} CODEENTRY;

static CODEENTRY	OpcodeTab[] = {
	{ OP_CLEAR, OP_Clear },
	{ OP_FUZZYFYINT, OP_FuzzifyInt },
	{ OP_DEFUZZYINT_COG, OP_DefuzzyIntCOG },
	{ OP_DEFUZZYINT_MHL, OP_DefuzzyIntMHL },
	{ OP_DEFUZZYINT_MHR, OP_DefuzzyIntMHR },
	{ OP_WEIGHT, OP_Weight },
	{ OP_MIN, OP_Min },
	{ OP_MAX, OP_Max },
	{ OP_END, NULL }
};

static unsigned		yResolution;
static BYTE			*pCode, *pCodeStart;
static BYTE			*pData, *pDataStart;
static BYTE			*pVariables;
static VALUE		AccuVal;
static size_t		InVarCount, OutVarCount;

static VALUE
fi( register VALUE x, register const BYTE * p )
{
	if( x < p[1] ) {
		if( x <= p[0] )	return 0;
		if( p[5] == 0 ) return (x - p[0]) * p[4];
		return (VALUE) (((((x - p[0]) * (unsigned) p[4]) >> (p[5] - 1)) + 1) >> 1);
	}
	if( x <= p[2] )	return (yResolution-1);
	if( x >= p[3] )	return 0;
	if( p[7] == 0 )	return (p[3] - x) * p[6];
	return (VALUE) (((((p[3] - x) * (unsigned) p[6]) >> (p[7] - 1)) + 1) >> 1);
}

/*
 *	OP_Clear
 */
static BYTE*	OP_Clear( BYTE * pMem )
{
	int		index;
	
	index = (int) *pMem++;
	pVariables[index] = 0;
	return pMem;
}

/*
 *	OP_FuzzifyInt
 */
static BYTE*	OP_FuzzifyInt( BYTE * pMem )
{
	int		varindex, memberindex;
	
	varindex = (int) *pMem++;
	memberindex = (int) *pMem++;
	pVariables[memberindex] = fi( pVariables[varindex], pData );
	pData += 8;
	return pMem;
}

/*
 *	OP_Weight
 */
static BYTE*	OP_Weight( BYTE * pMem )
{
	AccuVal = (VALUE) (((int) AccuVal * *pData++) / 256);
	return pMem;
}

/*
 *	OP_Min
 */
static BYTE*	OP_Min( BYTE * pMem )
{
	int		index, anzahl;
	VALUE	Value;
	
	anzahl = (int) *pMem++;
	AccuVal = yResolution - 1;
	while( anzahl-- > 0 ) {
		index = (int) *pMem++;
		Value = pVariables[index];
		if( Value < AccuVal )	AccuVal = Value;
	}
	return pMem;
}

/*
 *	OP_Max
 */
static BYTE*	OP_Max( BYTE * pMem )
{
	int		index, anzahl;
	
	index = (int) *pMem++;
	if( pVariables[index] < AccuVal )	pVariables[index] = AccuVal;
	return pMem;
}

/*
 *	OP_DefuzzyIntCOG
 */
static BYTE*	OP_DefuzzyIntCOG( BYTE * pMem )
{
	int		varindex, memberindex, anzahl;
	unsigned long	Zaehler = 0, Nenner = 0, Value;

	varindex = (int) *pMem++;
	anzahl = (int) *pMem++;
	while( anzahl-- > 0 ) {
		memberindex = (int) *pMem++;
		Value = pVariables[memberindex];
		Zaehler += Value * *pData++;
		Nenner += Value;
	}
	if( Nenner != 0 ) pVariables[varindex] = Zaehler / Nenner;
	return pMem;
}

/*
 *	OP_DefuzzyIntMHL
 */
static BYTE*	OP_DefuzzyIntMHL( BYTE * pMem )
{
	int		varindex, memberindex, anzahl;
	VALUE	Value, Position;
	VALUE	MaxVal, MaxPosition;
	BOOL	found = FALSE;

	varindex = (int) *pMem++;
	anzahl = (int) *pMem++;
	while( anzahl-- > 0 ) {
		memberindex = (int) *pMem++;
		Value = pVariables[memberindex];
		Position = *pData++;
		if( !found || Value > MaxVal ) {
			MaxVal = Value;
			found = TRUE;
			MaxPosition = Position;
		}
	}
	if( found ) pVariables[varindex] = MaxPosition;
	return pMem;
}

/*
 *	OP_DefuzzyIntMHR
 */
static BYTE*	OP_DefuzzyIntMHR( BYTE * pMem )
{
	int		varindex, memberindex, anzahl;
	VALUE	Value, Position;
	VALUE	MaxVal, MaxPosition;
	BOOL	found = FALSE;
	
	varindex = (int) *pMem++;
	anzahl = (int) *pMem++;
	while( anzahl-- > 0 ) {
		memberindex = (int) *pMem++;
		Value = pVariables[memberindex];
		Position = *pData++;
		if( !found || Value >= MaxVal ) {
			MaxVal = Value;
			found = TRUE;
			MaxPosition = Position;
		}
	}
	if( found ) pVariables[varindex] = MaxPosition;
	return pMem;
}

/*
 *	FuzzyEvaluate
 */
BOOL	FuzzyEvaluate( VALUE * InVars, VALUE * OutVars)
{
	OPCODE		Opcode;
	CODEENTRY	*CodeEntry;

	pCode = pCodeStart;
	pData = pDataStart;
	memcpy( &pVariables[0], InVars, InVarCount );
	while( (Opcode = (OPCODE) *pCode++) != OP_END ) {
		for( CodeEntry = OpcodeTab; CodeEntry->Opcode != OP_END; CodeEntry++ )
			if( CodeEntry->Opcode == Opcode )
				break;
		if( CodeEntry->CodeProc == NULL ) {
			printf( "Unbekannter Operator %02X\n", Opcode );
			return FALSE;
		}
		if( (pCode = CodeEntry->CodeProc( pCode )) == NULL )
			return FALSE;
	}
#if	0
	printf( "%3d %3d %3d\t%3d %3d %3d %3d\t",
		pVariables[4], pVariables[5], pVariables[6], pVariables[7],
		pVariables[8], pVariables[9], pVariables[10] );
//	printf( "%3d %3d %3d %3d\t", pVariables[11], pVariables[12], pVariables[13], pVariables[14] );
#endif
	memcpy( OutVars, &pVariables[InVarCount], OutVarCount );
	return TRUE;
}

/*
 *	ReadLong
 */
static BYTE	*ReadLong( BYTE * pMem, unsigned long * pLong )
{
	*pLong = (unsigned long) *pMem++;
	*pLong += ((unsigned long) *pMem++) << 8;
	*pLong += ((unsigned long) *pMem++) << 16;
	*pLong += ((unsigned long) *pMem++) << 24;
	return pMem;
}

/*
 *	FuzzyInit
 */
BOOL	FuzzyInit( BYTE * pMem, size_t MemLen )
{
	BYTE			*pMemStart = pMem;
	unsigned long	Magic, Version;
	unsigned long	CodeLen, DataLen, VarLen, PrivateLen;
	unsigned long	nInVars, nOutVars, yRes;

	pMem = ReadLong( pMem, &Magic );
	pMem = ReadLong( pMem, &Version );
	pMem = ReadLong( pMem, &CodeLen );
	pMem = ReadLong( pMem, &DataLen );
	pMem = ReadLong( pMem, &VarLen );
	pMem = ReadLong( pMem, &PrivateLen );
	pMem = ReadLong( pMem, &nInVars );
	pMem = ReadLong( pMem, &nOutVars );
	pMem = ReadLong( pMem, &yRes );
	pCodeStart = pMem;
	pDataStart = pCodeStart + CodeLen;
	pVariables = pDataStart + DataLen;
	if( HEADER_SIZE + CodeLen + DataLen + VarLen > MemLen ) {
		printf( "Der Speicher reicht nicht aus\n" );
		return FALSE;
	}
	InVarCount = (size_t) nInVars;
	OutVarCount = (size_t) nOutVars;
	yResolution = (unsigned) yRes;
	memset( pVariables, 0, (size_t) VarLen );
	return TRUE;
}

/*
 *	FuzzyTerminate
 */
void	FuzzyTerminate()
{
}
